/*-----------------------------------------------------------------------------
 * Complex Example: container implementation in C++
 *---------------------------------------------------------------------------*/

#if defined(WIN32) && !defined(__GNUC__)
// Turn off VC++ warnings on STL
#pragma warning(disable: 4786)
#pragma warning(disable: 4788)
#endif
 
#include <iostream>
#include <exception>
#include <string>
#include <vector>
#include <map>

#include <jni.h>
#include <stdlib.h>
#include <stdlib.h>
#include <windows.h>            // for DLL construction

#include "JniComplexExample.h"  // generated by 'javah -jni'
#include "jni_master.h"         // JNI encapsulation

using namespace std;

/*-----------------------------------------------------------------------------
 * The container is a singleton object, implemented as
 * multimap<string, JNIGlobalRef<jobject> *>, which has two thread-safe
 * access functions ('insert' and 'exportAllObjects').
 *
 * Values are stored by reference (as pointers), since it is not safe to have
 * auto-pointer-like structures in the map. Function 'insert' adds a given
 * object to the container, and function 'exportAllObjects' returns an array
 * of all the objects stored.
 *
 * To ensure code portability, thread-safety is realized by using JNI monitors.
 ----------------------------------------------------------------------------*/
class SampleContainer {
   static SampleContainer *instance;
   typedef multimap<string, JNIGlobalRef<jobject> *> MapOfObjects;

private:
   MapOfObjects mapOfObjects;		// the container implementation
   JNIGlobalRef<jobject> monitor;	// monitor (for critical sections)
   JNIEnv *_env;				    // the initial environment

   // Constructor: environment and monitor initialization
   SampleContainer(JNIEnv *env) :
	  _env(env), monitor(env, getMonitorObject(env))
   {}

   // Destructor: purging the map elements. 
   // All the global references are deleted explicitly.
   // The monitor object is released automatically through
   // the resource management mechanism.
   ~SampleContainer() {
	  MapOfObjects::iterator p;
	  for (p = mapOfObjects.begin(); p != mapOfObjects.end(); p++)
		 delete (*p).second;
   }

private:
   // allocating the monitor object
   static jobject getMonitorObject(JNIEnv *env) {
	  JNIClass objectClass(env, "java/lang/Object");
	  jmethodID constructorId = env->GetMethodID(objectClass, "<init>", "()V");
	  return env->NewObject(objectClass, constructorId);
   }
   
public:
   // Function 'getInstance()'.
   // The environment parameter facilitates the initialization mode.
   // If this parameter is 0, and no initialization has taken place before,
   // an exception will be thrown both in C++ and Java (observe that
   // dispatching a Java exception actually rises a flag in the Java
   // environment; this flag will be processed as the control returns from
   // the native code to Java. If this parameter is not zero, and no
   // initialization has taken place, the SampleContainer object is
   // initialized.
   // Note, that the initialization must be performed prior to any other
   // call to getInstance().
   static SampleContainer *getInstance(JNIEnv *env = 0) {
	  if (instance == 0) {
		 if (env == 0) {
			env->ThrowNew(JNIClass(env, "java/lang/Exception"),
						  "SampleContainer not initialized properly.");
			throw
			   new JNIException("SampleContainer not initialized properly.");
		 }
		 static JNIGlobalRef<jobject> initMonitor(env, getMonitorObject(env));

		 // Double-checked locking is used to provide correct initialization
		 JNIMonitor startCriticalSection(env, initMonitor);
		 if (instance == 0)
			instance = new SampleContainer(env);
	  }
	  return instance;
   }

   // inserting an object
   void insert(JNIEnv *env, jobject obj) {
	  // JNIMonitor is used to ensure integrity of the critical section.
	  // It is destroyed automatically as it goes out of scope.
	  JNIMonitor startCriticalSection(_env, monitor);

	  // Retrieve the "name" field of the object, create a global reference to
	  // it, then insert a pointer to this reference into the container.
	  // A global reference is required, since otherwise Java
	  // garbage collector may destroy the object prematurely.
	  JNIStringUTFChars str(env, obj, "name");
	  JNIGlobalRef<jobject> *ref = new JNIGlobalRef<jobject>(env, obj);
	  mapOfObjects.insert(make_pair(str.asString(), ref));
   }

   // Exporting all the collected objects as a vector (this vector is
   // inherently sorted, as the objects are extracted from a map).
   // Observe that it's impossible to use the STL 'copy()' here, since 'vector'
   // and 'multimap' iterators have a different structure.
   vector<JNIGlobalRef<jobject> *> exportAllObjects() {
	  JNIMonitor startCriticalSection(_env, monitor);
	  vector<JNIGlobalRef<jobject> *> result(mapOfObjects.size(), 0);
	  MapOfObjects::iterator p;
	  vector<JNIGlobalRef<jobject> *>::iterator q;
	  for (p = mapOfObjects.begin(), q = result.begin();
		   p != mapOfObjects.end(); p++, q++)
		 *q = (*p).second;
	  return result;
   }	  
};

// Singleton instance
SampleContainer *SampleContainer::instance = 0;

/*-----------------------------------------------------------------------------
 * Implementation of native calls
 *---------------------------------------------------------------------------*/

// Initialization: creating a singleton instance
// (explicit call with the environment parameter)
JNIEXPORT void JNICALL Java_JniComplexExample_init_1native_1resources
  (JNIEnv *env, jclass clazz)
{
   SampleContainer::getInstance(env);
}

// Inserting an object
JNIEXPORT void JNICALL Java_JniComplexExample_register_1object
  (JNIEnv *env, jclass clazz, jobject obj)
{
   SampleContainer::getInstance()->insert(env, obj);
}

// Exporting all objects.
// This is the only function where we need the class of the object,
// in order to create an array.
JNIEXPORT jobjectArray JNICALL Java_JniComplexExample_recall_1objects
  (JNIEnv *env, jclass clazz)
{
   // Obtain the vector of global references
   vector<JNIGlobalRef<jobject> *> allObjects =
	  SampleContainer::getInstance()->exportAllObjects();
   // Create an output array of type 'NameWithInfo[]'
   JNIClass objectClass(env, "NameWithInfo");
   jobjectArray result =
	  env->NewObjectArray(allObjects.size(), objectClass, 0);
   // Export the objects, then return the result
   for (int i = 0; i < allObjects.size(); i++)
	  env->SetObjectArrayElement(result, i,
								 env->NewGlobalRef(*allObjects[i]));
   return result;
}

